SET NAMES utf8mb4;
SET time_zone = '-03:00';

-- ===============================
-- Tabelas de autenticação
-- ===============================
CREATE TABLE IF NOT EXISTS admin_users (
  id INT AUTO_INCREMENT PRIMARY KEY,
  username VARCHAR(60) NOT NULL UNIQUE,
  password_hash VARCHAR(255) NOT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS customers (
  id INT AUTO_INCREMENT PRIMARY KEY,
  full_name VARCHAR(160) NOT NULL,
  phone VARCHAR(40) NOT NULL UNIQUE,
  password_hash VARCHAR(255) NOT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  last_login_at DATETIME NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===============================
-- Catálogo (itens, cores, kits)
-- ===============================
CREATE TABLE IF NOT EXISTS items (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(160) NOT NULL,
  image_path VARCHAR(255) NULL,
  category ENUM('panel','table','tray','jar','mini_cake','cake_stand','accessory') NOT NULL,
  price DECIMAL(10,2) NOT NULL DEFAULT 0,
  active TINYINT(1) DEFAULT 1,
  UNIQUE KEY uniq_item_name (name)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS item_variants (
  id INT AUTO_INCREMENT PRIMARY KEY,
  item_id INT NOT NULL,
  variant_name VARCHAR(80) NOT NULL,
  stock_qty INT NOT NULL DEFAULT 0,
  active TINYINT(1) DEFAULT 1,
  UNIQUE KEY uniq_item_variant (item_id, variant_name),
  KEY idx_item (item_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS kits (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(160) NOT NULL,
  price DECIMAL(10,2) NOT NULL DEFAULT 0,
  photo_path VARCHAR(255) NULL,
  active TINYINT(1) DEFAULT 1,
  UNIQUE KEY uniq_kit_name (name)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS kit_items (
  id INT AUTO_INCREMENT PRIMARY KEY,
  kit_id INT NOT NULL,
  item_id INT NOT NULL,
  qty INT NOT NULL DEFAULT 1,
  customizable TINYINT(1) NOT NULL DEFAULT 0,
  UNIQUE KEY uniq_kit_item (kit_id, item_id),
  KEY idx_kit (kit_id),
  KEY idx_item (item_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===============================
-- Pedidos
-- ===============================
CREATE TABLE IF NOT EXISTS orders (
  id INT AUTO_INCREMENT PRIMARY KEY,
  code VARCHAR(32) NOT NULL UNIQUE,
  customer_id INT NOT NULL,
  kit_id INT NOT NULL,
  event_date DATE NOT NULL,
  theme_text VARCHAR(160) NOT NULL,
  theme_id INT NULL,
  pickup_mode ENUM('client','app') NOT NULL DEFAULT 'client',
  return_mode ENUM('client','app') NOT NULL DEFAULT 'client',
  pickup_at DATETIME NOT NULL,
  return_at DATETIME NOT NULL,

  kit_price DECIMAL(10,2) NOT NULL DEFAULT 0,
  total_value DECIMAL(10,2) NOT NULL DEFAULT 0,
  deposit_required DECIMAL(10,2) NOT NULL DEFAULT 0,
  payment_status ENUM('unpaid','deposit_paid','paid') NOT NULL DEFAULT 'unpaid',
  paid_value DECIMAL(10,2) NOT NULL DEFAULT 0,
  paid_at DATETIME NULL,

  status ENUM('pending','confirmed','picked_up','returned','canceled') NOT NULL DEFAULT 'pending',
  notes TEXT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

  KEY idx_customer (customer_id),
  KEY idx_pickup (pickup_at),
  KEY idx_event (event_date),
  KEY idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS order_customizations (
  id INT AUTO_INCREMENT PRIMARY KEY,
  order_id INT NOT NULL,
  item_id INT NOT NULL,
  variant_id INT NOT NULL,
  qty INT NOT NULL DEFAULT 1,
  KEY idx_order (order_id),
  KEY idx_variant (variant_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===============================
-- Seed: admin padrão
-- ===============================
INSERT INTO admin_users (username, password_hash)
VALUES ('admin', '$2y$10$IHHxIfEH6pXlggEVJZzGduKuDcenxdb.cld2jvcs1R/IKnt/LtWF2')
ON DUPLICATE KEY UPDATE password_hash=VALUES(password_hash);

-- ===============================
-- Seed: Itens, cores, kits e composição
-- ===============================
START TRANSACTION;

INSERT INTO items (name, category, price, active) VALUES
('Painel', 'panel', 30.00, 1),
('Boleira', 'cake_stand', 0.00, 1),
('Bandeja', 'tray', 0.00, 1),
('Jarro', 'jar', 0.00, 1),
('Mini cake', 'mini_cake', 0.00, 1),
('Arco de bolas', 'accessory', 20.00, 1),
('Display', 'accessory', 0.00, 1),

('Mesa X', 'table', 25.00, 1),
('Mesa balcão P', 'table', 20.00, 1),
('Mesa balcão G', 'table', 30.00, 1),
('Mesa redonda', 'table', 30.00, 1),
('Mesa cavalete 1,20m', 'table', 40.00, 1),

('Escada lembrancinha P', 'accessory', 10.00, 1),
('Escada lembrancinha M', 'accessory', 14.00, 1),
('Escada lembrancinha G', 'accessory', 17.00, 1),
('Personalização adesiva', 'accessory', 10.00, 1)
ON DUPLICATE KEY UPDATE
  category = VALUES(category),
  price = VALUES(price),
  active = VALUES(active);

SET @item_bandeja   := (SELECT id FROM items WHERE name='Bandeja'   LIMIT 1);
SET @item_jarro     := (SELECT id FROM items WHERE name='Jarro'     LIMIT 1);
SET @item_minicake  := (SELECT id FROM items WHERE name='Mini cake' LIMIT 1);
SET @item_boleira   := (SELECT id FROM items WHERE name='Boleira'   LIMIT 1);

INSERT INTO item_variants (item_id, variant_name, stock_qty, active) VALUES
(@item_bandeja,  'Branco', 5, 1),
(@item_bandeja,  'Rosa', 5, 1),
(@item_bandeja,  'Dourado', 5, 1),
(@item_bandeja,  'Prata', 5, 1),
(@item_bandeja,  'Preto', 5, 1),

(@item_jarro,    'Branco', 5, 1),
(@item_jarro,    'Rosa', 5, 1),
(@item_jarro,    'Dourado', 5, 1),
(@item_jarro,    'Prata', 5, 1),
(@item_jarro,    'Preto', 5, 1),

(@item_minicake, 'Branco', 5, 1),
(@item_minicake, 'Rosa', 5, 1),
(@item_minicake, 'Dourado', 5, 1),
(@item_minicake, 'Prata', 5, 1),
(@item_minicake, 'Preto', 5, 1),

(@item_boleira,  'Branco', 5, 1),
(@item_boleira,  'Rosa', 5, 1),
(@item_boleira,  'Dourado', 5, 1),
(@item_boleira,  'Prata', 5, 1),
(@item_boleira,  'Preto', 5, 1)
ON DUPLICATE KEY UPDATE
  stock_qty = VALUES(stock_qty),
  active = VALUES(active);

INSERT INTO kits (name, price, active) VALUES
('Pocket Party Básico', 49.99, 1),
('Pocket Party Médio', 59.99, 1),
('Pocket Party Só um bolinho', 55.99, 1),
('Pocket Party Premium', 79.99, 1),
('Pocket Party Lounge', 84.99, 1),
('Pocket Party Retrô', 69.99, 1),
('Pocket Party Estilo Chic', 79.99, 1),
('Pocket Party Clássico', 94.99, 1),
('Pocket Party Elegance', 104.99, 1),
('Pocket Party Piquenique', 114.99, 1),
('Pocket Party Interativo', 109.99, 1)
ON DUPLICATE KEY UPDATE
  price = VALUES(price),
  active = VALUES(active);

-- itens
SET @it_painel   := (SELECT id FROM items WHERE name='Painel' LIMIT 1);
SET @it_boleira  := (SELECT id FROM items WHERE name='Boleira' LIMIT 1);
SET @it_bandeja  := (SELECT id FROM items WHERE name='Bandeja' LIMIT 1);
SET @it_jarro    := (SELECT id FROM items WHERE name='Jarro' LIMIT 1);
SET @it_minicake := (SELECT id FROM items WHERE name='Mini cake' LIMIT 1);
SET @it_arco     := (SELECT id FROM items WHERE name='Arco de bolas' LIMIT 1);
SET @it_display  := (SELECT id FROM items WHERE name='Display' LIMIT 1);

SET @it_mesa_x   := (SELECT id FROM items WHERE name='Mesa X' LIMIT 1);
SET @it_balcao_p := (SELECT id FROM items WHERE name='Mesa balcão P' LIMIT 1);
SET @it_balcao_g := (SELECT id FROM items WHERE name='Mesa balcão G' LIMIT 1);
SET @it_cavalete := (SELECT id FROM items WHERE name='Mesa cavalete 1,20m' LIMIT 1);

-- kits
SET @k1  := (SELECT id FROM kits WHERE name='Pocket Party Básico' LIMIT 1);
SET @k2  := (SELECT id FROM kits WHERE name='Pocket Party Médio' LIMIT 1);
SET @k3  := (SELECT id FROM kits WHERE name='Pocket Party Só um bolinho' LIMIT 1);
SET @k4  := (SELECT id FROM kits WHERE name='Pocket Party Premium' LIMIT 1);
SET @k5  := (SELECT id FROM kits WHERE name='Pocket Party Lounge' LIMIT 1);
SET @k6  := (SELECT id FROM kits WHERE name='Pocket Party Retrô' LIMIT 1);
SET @k7  := (SELECT id FROM kits WHERE name='Pocket Party Estilo Chic' LIMIT 1);
SET @k8  := (SELECT id FROM kits WHERE name='Pocket Party Clássico' LIMIT 1);
SET @k9  := (SELECT id FROM kits WHERE name='Pocket Party Elegance' LIMIT 1);
SET @k10 := (SELECT id FROM kits WHERE name='Pocket Party Piquenique' LIMIT 1);
SET @k11 := (SELECT id FROM kits WHERE name='Pocket Party Interativo' LIMIT 1);

DELETE FROM kit_items WHERE kit_id IN (@k1,@k2,@k3,@k4,@k5,@k6,@k7,@k8,@k9,@k10,@k11);

-- Kit 1
INSERT INTO kit_items (kit_id, item_id, qty, customizable) VALUES
(@k1, @it_painel, 1, 0),
(@k1, @it_boleira, 1, 1),
(@k1, @it_bandeja, 1, 1),
(@k1, @it_jarro, 1, 1);

-- Kit 2
INSERT INTO kit_items (kit_id, item_id, qty, customizable) VALUES
(@k2, @it_painel, 1, 0),
(@k2, @it_boleira, 1, 1),
(@k2, @it_bandeja, 2, 1),
(@k2, @it_jarro, 1, 1),
(@k2, @it_minicake, 1, 1);

-- Kit 3
INSERT INTO kit_items (kit_id, item_id, qty, customizable) VALUES
(@k3, @it_painel, 1, 0),
(@k3, @it_jarro, 1, 1),
(@k3, @it_arco, 1, 0);

-- Kit 4
INSERT INTO kit_items (kit_id, item_id, qty, customizable) VALUES
(@k4, @it_painel, 1, 0),
(@k4, @it_boleira, 1, 1),
(@k4, @it_jarro, 1, 1),
(@k4, @it_bandeja, 2, 1),
(@k4, @it_minicake, 1, 1),
(@k4, @it_arco, 1, 0);

-- Kit 5
INSERT INTO kit_items (kit_id, item_id, qty, customizable) VALUES
(@k5, @it_painel, 1, 0),
(@k5, @it_boleira, 1, 1),
(@k5, @it_bandeja, 2, 1),
(@k5, @it_jarro, 1, 1),
(@k5, @it_minicake, 1, 1),
(@k5, @it_arco, 1, 0),
(@k5, @it_display, 1, 0);

-- Kit 6
INSERT INTO kit_items (kit_id, item_id, qty, customizable) VALUES
(@k6, @it_painel, 1, 0),
(@k6, @it_boleira, 1, 1),
(@k6, @it_bandeja, 1, 1),
(@k6, @it_jarro, 1, 1),
(@k6, @it_balcao_p, 1, 0);

-- Kit 7
INSERT INTO kit_items (kit_id, item_id, qty, customizable) VALUES
(@k7, @it_painel, 1, 0),
(@k7, @it_boleira, 1, 1),
(@k7, @it_bandeja, 2, 1),
(@k7, @it_jarro, 1, 1),
(@k7, @it_minicake, 1, 1),
(@k7, @it_mesa_x, 1, 0);

-- Kit 8
INSERT INTO kit_items (kit_id, item_id, qty, customizable) VALUES
(@k8, @it_painel, 1, 0),
(@k8, @it_boleira, 1, 1),
(@k8, @it_bandeja, 2, 1),
(@k8, @it_jarro, 1, 1),
(@k8, @it_minicake, 1, 1),
(@k8, @it_mesa_x, 1, 0),
(@k8, @it_arco, 1, 0),
(@k8, @it_display, 1, 0);

-- Kit 9
INSERT INTO kit_items (kit_id, item_id, qty, customizable) VALUES
(@k9, @it_painel, 1, 0),
(@k9, @it_boleira, 1, 1),
(@k9, @it_jarro, 1, 1),
(@k9, @it_bandeja, 2, 1),
(@k9, @it_minicake, 2, 1),
(@k9, @it_balcao_g, 1, 0),
(@k9, @it_arco, 1, 0),
(@k9, @it_display, 1, 0);

-- Kit 10
INSERT INTO kit_items (kit_id, item_id, qty, customizable) VALUES
(@k10, @it_painel, 1, 0),
(@k10, @it_boleira, 1, 1),
(@k10, @it_jarro, 2, 1),
(@k10, @it_minicake, 2, 1),
(@k10, @it_bandeja, 2, 1),
(@k10, @it_cavalete, 1, 0),
(@k10, @it_arco, 1, 0),
(@k10, @it_display, 1, 0);

-- Kit 11
INSERT INTO kit_items (kit_id, item_id, qty, customizable) VALUES
(@k11, @it_painel, 1, 0),
(@k11, @it_boleira, 1, 1),
(@k11, @it_bandeja, 2, 1),
(@k11, @it_minicake, 2, 1),
(@k11, @it_jarro, 2, 1),
(@k11, @it_balcao_g, 1, 0),
(@k11, @it_arco, 1, 0),
(@k11, @it_display, 1, 0);

COMMIT;
