<?php
declare(strict_types=1);

/**
 * Salva uma imagem enviada via POST (input type=file) no diretório /uploads.
 * Retorna o caminho relativo (ex: uploads/kit_20260130_xxx.webp) ou null se não foi enviada.
 */
function save_uploaded_image(string $field, string $prefix='img_'): ?string {
  if (empty($_FILES[$field]) || !is_array($_FILES[$field])) return null;

  $err = $_FILES[$field]['error'] ?? UPLOAD_ERR_NO_FILE;
  if ($err === UPLOAD_ERR_NO_FILE) return null;
  if ($err !== UPLOAD_ERR_OK) {
    throw new Exception("Falha no upload (código $err).");
  }

  $tmp  = $_FILES[$field]['tmp_name'] ?? '';
  $name = $_FILES[$field]['name'] ?? '';
  if (!$tmp || !is_uploaded_file($tmp)) {
    throw new Exception("Upload inválido.");
  }

  $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
  $allowed = ['jpg','jpeg','png','webp'];
  if (!in_array($ext, $allowed, true)) {
    throw new Exception("Formato inválido. Use JPG, PNG ou WEBP.");
  }

  // valida mime real
  $finfo = new finfo(FILEINFO_MIME_TYPE);
  $mime = $finfo->file($tmp);
  $okMime = ['image/jpeg','image/png','image/webp'];
  if (!in_array($mime, $okMime, true)) {
    throw new Exception("Arquivo não é uma imagem válida.");
  }

  // cria pasta
  $dir = __DIR__ . '/../uploads';
  if (!is_dir($dir)) {
    if (!mkdir($dir, 0755, true) && !is_dir($dir)) {
      throw new Exception("Não foi possível criar a pasta uploads.");
    }
  }

  $safe = $prefix . date('Ymd_His') . '_' . bin2hex(random_bytes(6)) . '.' . $ext;
  $dest = $dir . '/' . $safe;

  if (!move_uploaded_file($tmp, $dest)) {
    throw new Exception("Falha ao salvar a imagem.");
  }

  return 'uploads/' . $safe;
}
