<?php
declare(strict_types=1);

/** Quantidade reservada de uma cor (variant_id) no período. */
function reserved_qty_for_variant(PDO $pdo, int $variant_id, string $pickup_at, string $return_at): int {
  $sql = "
    SELECT COALESCE(SUM(oc.qty),0) AS q
    FROM order_customizations oc
    JOIN orders o ON o.id = oc.order_id
    WHERE oc.variant_id = ?
      AND o.status IN ('pending','confirmed','picked_up')
      AND o.pickup_at < ?
      AND o.return_at > ?
  ";
  $st = $pdo->prepare($sql);
  $st->execute([$variant_id, $return_at, $pickup_at]);
  $r = $st->fetch();
  return (int)($r['q'] ?? 0);
}

/** Variantes disponíveis para um item no período. */
function available_variants_for_item(PDO $pdo, int $item_id, int $qty_needed, string $pickup_at, string $return_at): array {
  $st = $pdo->prepare("SELECT id, variant_name, stock_qty FROM item_variants WHERE item_id=? AND active=1 ORDER BY variant_name ASC");
  $st->execute([$item_id]);
  $vars = $st->fetchAll();

  $out = [];
  foreach ($vars as $v) {
    $reserved = reserved_qty_for_variant($pdo, (int)$v['id'], $pickup_at, $return_at);
    $avail = (int)$v['stock_qty'] - $reserved;
    if ($avail >= $qty_needed) {
      $out[] = ['id'=>(int)$v['id'], 'name'=>(string)$v['variant_name'], 'avail'=>$avail];
    }
  }
  return $out;
}

/** Valida escolhas de cor do kit. */
function validate_kit_choices(PDO $pdo, int $kit_id, array $choices, string $pickup_at, string $return_at): array {
  $st = $pdo->prepare("SELECT item_id, qty, customizable FROM kit_items WHERE kit_id=?");
  $st->execute([$kit_id]);
  $rows = $st->fetchAll();
  if (!$rows) return [false, "Kit inválido."];

  foreach ($rows as $r) {
    $item_id = (int)$r['item_id'];
    $qty_needed = (int)$r['qty'];
    $custom = (int)$r['customizable'] === 1;

    if ($custom) {
      $variant_id = isset($choices[$item_id]) ? (int)$choices[$item_id] : 0;
      if ($variant_id <= 0) return [false, "Selecione a cor de todos os itens obrigatórios."];

      $ck = $pdo->prepare("SELECT stock_qty FROM item_variants WHERE id=? AND item_id=? AND active=1 LIMIT 1");
      $ck->execute([$variant_id, $item_id]);
      $v = $ck->fetch();
      if (!$v) return [false, "Cor inválida selecionada."];

      $reserved = reserved_qty_for_variant($pdo, $variant_id, $pickup_at, $return_at);
      $avail = (int)$v['stock_qty'] - $reserved;

      if ($avail < $qty_needed) {
        $alts = available_variants_for_item($pdo, $item_id, $qty_needed, $pickup_at, $return_at);
        $names = array_map(fn($a)=>$a['name'], $alts);
        return [false, "Cor indisponível para um item do kit. Outras cores disponíveis: ".implode(', ', $names)];
      }
    }
  }
  return [true, "OK"];
}
